<#
    Copyright (c) 2023. Cloud Software Group, Inc. All Rights Reserved.
.SYNOPSIS 
    Citrix Optimization Engine helps to optimize operating system to run better with XenApp or XenDesktop solutions.

.DESCRIPTION
    Citrix Optimization Engine helps to optimize operating system to run better with XenApp or XenDesktop solutions. This script can run in three different modes - Analyze, Execute and Rollback. Each execution will automatically generate an XML file with a list of performed actions (stored under .\Logs folder) that can be used to rollback the changes applied. 

.PARAMETER Source
    Source XML file that contains the required configuration. Typically located under .\Templates folder. This file provides instructions that CTXOE can process. Template can be specified with or without .xml extension and can be full path or just a filename. If you specify only filename, template must be located in .\Templates folder.
    If -Source or -Template is not specified, Optimizer will automatically try to detect the best suitable template. It will look in .\Templates folder for file called <templateprefix>_<OS>_<build>. See help for -templateprefix to learn more about using your own custom templates.

.PARAMETER TemplatePrefix
    When -Source or -Template parameter is not specified, Optimizer will try to find the best matching template automatically. By default, it is looking for templates that start with "Citrix_Windows" and are provided by Citrix as part of default Optimizer build. If you would like to use your own templates with auto-select, you can override the default templates prefix. 
    For example if your template is called My_Windows_10_1809.xml, use '-TemplatePrefix "My_Windows"' to automatically select your templates based on current operating system and build.

.PARAMETER Mode
    CTXOE supports three different modes:
        Analyze - Do not apply any changes, only show the recommended changes.
        Execute - Apply the changes to the operating system.
        Rollback - Revert the applied changes. Requires a valid XML backup from the previously run Execute phase. This file is usually called Execute_History.xml.

    WARNING: Rollback mode cannot restore applications that have been removed. If you are planning to remove UWP applications and want to be able to recover them, use snapshots instead of rollback mode.

.PARAMETER IgnoreConditions
    When you use -IgnoreConditions switch, all conditions are skipped and optimizations are applied without any environments tests. This is used mostly for troubleshooting and is not recommended for normal environments.

.PARAMETER Groups
    Array that allows you to specify which groups to process from a specified source file.

.PARAMETER OutputLogFolder
    The location where to save all generated log files. This will replace an automatically generated folder .\Logs and is typically used with ESD solutions like SCCM.

.PARAMETER OutputXml
    The location where the output XML should be saved. The XML with results is automatically saved under .\Logs folder, but you can optionally specify also other location. This argument can be used together with -OutputHtml.

.PARAMETER OutputHtml
    The location where the output HTML report should be saved. The HTML with results is automatically saved under .\Logs folder, but you can optionally specify another location. This argument can be used together with -OutputXml.

.PARAMETER OptimizerUI
    Parameter used by Citrix Optimizer Tool UI to retrieve information from optimization engine. For internal use only.

.EXAMPLE
    .\CtxOptimizerEngine.ps1 -Source C:\Temp\Win10.xml -Mode Analyze
    Process all entries in Win10.xml file and display the recommended changes. Changes are not applied to the system.

.EXAMPLE
    .\CtxOptimizerEngine.ps1 -Source C:\Temp\Win10.xml -Mode Execute
    Process all entries from Win10.xml file. These changes are applied to the operating system.

.EXAMPLE
    .\CtxOptimizerEngine.ps1 -Source C:\Temp\Win10.xml -Mode Execute -Groups "DisableServices", "RemoveApplications"
    Process entries from groups "Disable Services" and "Remove built-in applications" in Win10.xml file. These changes are applied to the operating system.

.EXAMPLE
    .\CtxOptimizerEngine.ps1 -Source C:\Temp\Win10.xml -Mode Execute -OutputXml C:\Temp\Rollback.xml
    Process all entries from Win10.xml file. These changes are applied to the operating system. Save the rollback instructions in the file rollback.xml.

.EXAMPLE
    .\CtxOptimizerEngine.ps1 -Source C:\Temp\Rollback.xml -Mode Rollback
    Revert all changes from the file rollback.xml.

.NOTES
    Author: Martin Zugec
    Date:   February 17, 2017

.LINK
    https://support.citrix.com/article/CTX224676
#>

#Requires -Version 2

Param (
    [Alias("Template")]
    [System.String]$Source,

    [ValidateSet('analyze','execute','rollback')]

    [System.String]$Mode = "Analyze",

    [Array]$Groups,

    [String]$OutputLogFolder,

    [String]$OutputHtml,

    [String]$OutputXml,

    [Switch]$OptimizerUI,

    [Switch]$IgnoreConditions,

    [String]$TemplatePrefix
)

[String]$EngineVersion = "2.9";
# Retrieve friendly OS name (e.g. Winodws 10 Pro)
[String]$m_OSName = (Get-WmiObject Win32_OperatingSystem).Caption;
# If available, retrieve a build number (yymm like 1808). This is used on Windows Server 2016 and Windows 10, but is not used on older operating systems and is optional
[String]$m_OSBuild = $(Get-ItemProperty 'HKLM:\SOFTWARE\Microsoft\Windows NT\CurrentVersion' -Name ReleaseID -ErrorAction SilentlyContinue | Select-Object -ExpandProperty ReleaseID);

Write-Host "------------------------------"
Write-Host "| Citrix Optimization Engine |"
Write-Host "| Version $EngineVersion                |"
Write-Host "------------------------------"
Write-Host

Write-Host "Running in " -NoNewline
Write-Host -ForegroundColor Yellow $Mode -NoNewLine
Write-Host " mode"

# Error handling. We want Citrix Optimizer Tool to abort on any error, so error action preference is set to "Stop".
# The problem with this approach is that if Optimizer is called from another script, "Stop" instruction will apply to that script as well, so failure in Optimizer engine will abort calling script(s).
# As a workaround, instead of terminating the script, Optimizer has a global error handling procedure that will restore previous setting of ErrorActionPreference and properly abort the execution.
$OriginalErrorActionPreferenceValue = $ErrorActionPreference;
$ErrorActionPreference = "Stop";

Trap {
    Write-Host "Citrix Optimizer Tool engine has encountered a problem and will now terminate";
    $ErrorActionPreference = $OriginalErrorActionPreferenceValue;
    Write-Error $_;

    # Update $Run_Status with error encountered and save output XML file.
    If ($Run_Status) {
        $Run_Status.run_successful = $False.ToString();
        $Run_Status.run_details = "Error: $_";
        $Run_Status.time_end = [DateTime]::Now.ToString('yyyy-MM-dd_HH-mm-ss') # Saving DateTime in predefined format. This is required, since we don't know the target localization settings and want to make sure that UI and engine can communicate in same language.
        $PackDefinitionXml.Save($ResultsXml);
    }

    Return $False;
}

# Create enumeration for PluginMode. Enumeration cannot be used in the param() section, as that would require a DynamicParam on a script level.
[String]$PluginMode = $Mode;

# Just in case if previous run failed, make sure that all modules are reloaded
Remove-Module CTXOE*;

# Create $CTXOE_Main variable that defines folder where the script is located. If code is executed manually (copy & paste to PowerShell window), current directory is being used
If ($MyInvocation.MyCommand.Path -is [Object]) {
    [string]$Global:CTXOE_Main = $(Split-Path -Parent $MyInvocation.MyCommand.Path);
} Else {
    [string]$Global:CTXOE_Main = $(Get-Location).Path;
}

# Create Logs folder if it doesn't exists
If ($OutputLogFolder.Length -eq 0) {
    $Global:CTXOE_LogFolder = "$CTXOE_Main\Logs\$([DateTime]::Now.ToString('yyyy-MM-dd_HH-mm-ss'))"
} Else {
    $Global:CTXOE_LogFolder = $OutputLogFolder;
}

If ($(Test-Path "$CTXOE_LogFolder") -eq $false) {
    Write-Host "Creating Logs folder $(Split-Path -Leaf $CTXOE_LogFolder)"
    MkDir $CTXOE_LogFolder | Out-Null
}

# Report the location of log folder to UI
If ($OptimizerUI) {
    $LogFolder = New-Object -TypeName PSObject
    $LogFolder.PSObject.TypeNames.Insert(0,"logfolder")
    $LogFolder | Add-Member -MemberType NoteProperty -Name Location -Value $CTXOE_LogFolder
    Write-Output $LogFolder
}

# Initialize debug log (transcript). PowerShell ISE doesn't support transcriptions at the moment.
# Previously, we tried to determine if current host supports transcription or not, however this functionality is broken since PowerShell 4.0. Using Try/Catch instead.
Write-Host "Starting session log"
Try {
    $CTXOE_DebugLog = "$CTXOE_LogFolder\Log_Debug_CTXOE.log"
    Start-Transcript -Append -Path "$CTXOE_DebugLog" | Out-Null
} Catch { Write-Host "An exception happened when starting transcription: $_" -ForegroundColor Red }

# Check if user is administrator
Write-Host "Checking permissions"
If (-not ([Security.Principal.WindowsPrincipal] [Security.Principal.WindowsIdentity]::GetCurrent()).IsInRole([Security.Principal.WindowsBuiltInRole] "Administrator")) {
    Throw "You must be administrator in order to execute this script"
}

# Check if template name has been provided. If not, try to detect proper template automatically
If ($Source.Length -eq 0 -or $Source -eq "AutoSelect") {
    Write-Host "Template not specified, turning on auto-select mode";
    
    # Multiple template prefixes can be used - users can have their own custom templates.
    [array]$m_TemplatePrefixes = @();
    If ($TemplatePrefix.Length -gt 0) {
        Write-Host "Custom template prefix detected: $TemplatePrefix";
        $m_TemplatePrefixes += $TemplatePrefix;
    }
    $m_TemplatePrefixes += "Citrix_Windows";
    
    # Strip the description, keep only numbers. Special processing is required to include "R2" versions. Result of this regex is friendly version number (7, 10 or '2008 R2' for example)
    [String]$m_TemplateNameOSVersion = $([regex]"([0-9])+\sR([0-9])+|[(0-9)]+").Match($m_OSName).Captures[0].Value.Replace(" ", "");
    
    # Go through all available template prefixes, starting with custom prefix. Default Citrix prefix is used as a last option
    ForEach ($m_TemplateNamePrefix in $m_TemplatePrefixes) {

        Write-Host "Trying to find matching templates for prefix $m_TemplateNamePrefix"

        # If this is server OS, include "Server" in the template name. If this is client, don't do anything. While we could include _Client in the template name, it just looks weird.
        If ((Get-ItemProperty 'HKLM:\SOFTWARE\Microsoft\Windows NT\CurrentVersion' -Name InstallationType).InstallationType -eq "Server") {
            $m_TemplateNamePrefix += "_Server";
            If ($TemplatePrefix.Length -gt 0) {$m_TemplateNameCustomPrefix += "_Server";}
        }

        # First, we try to find if template for current OS and build is available. If not, we tried to find last build for the same OS version. If that is not available, we finally check for generic version of template (not build specific)
        If (Test-Path -Path "$CTXOE_Main\Templates\$(($m_TemplateNamePrefix) + '_' + ($m_TemplateNameOSVersion) + '_' + ($m_OSBuild)).xml") {
            Write-Host "Template detected - using optimal template for current version and build";
            $Source = "$CTXOE_Main\Templates\$(($m_TemplateNamePrefix) + '_' + ($m_TemplateNameOSVersion) + '_' + ($m_OSBuild)).xml";
            Break;
        } Else {
            Write-Host "Preferred template $(($m_TemplateNamePrefix) + '_' + ($m_TemplateNameOSVersion) + '_' + ($m_OSBuild)) was not found, using fallback mode"
            [array]$m_PreviousBuilds = Get-ChildItem -Path "$CTXOE_Main\Templates" -Filter $($m_TemplateNamePrefix + '_' + ($m_TemplateNameOSVersion) + '_*');
            # Older versions of PowerShell (V2) will automatically delect object instead of initiating an empty array.
            If ($m_PreviousBuilds -isnot [Object] -or $m_PreviousBuilds.Count -eq 0) {
                If (Test-Path -Path "$CTXOE_Main\Templates\$(($m_TemplateNamePrefix) + '_' + ($m_TemplateNameOSVersion)).xml") {
                    Write-Host "Template detected - using generic template for OS version";
                    $Source = "$CTXOE_Main\Templates\$(($m_TemplateNamePrefix) + '_' + ($m_TemplateNameOSVersion)).xml";
                    Break;
                }
            } Else {
                Write-Host "Template detected - using previous OS build";
                $Source = "$CTXOE_Main\Templates\$($m_PreviousBuilds | Sort-Object Name | Select-Object -ExpandProperty Name -Last 1)";
                Break;
            }
        }
    
    }

    If ($Source.Length -eq 0 -or $Source -eq "AutoSelect")  {Throw "Auto-detection of template failed, no suitable template has been found"}

}

# Check if -Source is a fullpath or just name of the template. If it's just the name, expand to a fullpath.
If (-not $Source.Contains("\")) {
    If (-not $Source.ToLower().EndsWith(".xml")) {
         $Source = "$Source.xml";
    }

    $Source = "$CTXOE_Main\Templates\$Source";
}

# Specify the default location of output XML
[String]$ResultsXml = "$CTXOE_LogFolder\$($PluginMode)_History.xml"
If ($OutputHtml.Length -eq 0) {
    [String]$OutputHtml = "$CTXOE_LogFolder\$($PluginMode)_History.html"
}

Write-Host
Write-Host "Processing definition file $Source"
[Xml]$PackDefinitionXml = Get-Content $Source

# Try to find if this template has been executed before. If <runstatus /> is present, move it to history (<previousruns />). This is used to store all previous executions of this template.
If ($PackDefinitionXml.root.run_status) {
    # Check if <previousruns /> exists. If not, create a new one.
    If (-not $PackDefinitionXml.root.previousruns) {
        $PackDefinitionXml.root.AppendChild($PackDefinitionXml.CreateElement("previousruns")) | Out-Null;
    }

    $PackDefinitionXml.root.Item("previousruns").AppendChild($PackDefinitionXml.root.run_status) | Out-Null;
}

# Create new XML element to store status of the execution.
[System.Xml.XmlElement]$Run_Status = $PackDefinitionXml.root.AppendChild($PackDefinitionXml.ImportNode($([Xml]"<run_status><run_mode /><time_start /><time_end /><entries_total /><entries_success /><entries_failed /><run_successful /><run_details /><engineversion /><targetos /><targetcomputer /></run_status>").DocumentElement, $True));
$Run_Status.run_successful = $False.ToString();
$Run_Status.run_mode = $PluginMode;
$Run_Status_Default_Message = "Run started, but never finished";
$Run_Status.run_details = $Run_Status_Default_Message;
$Run_Status.time_start = [DateTime]::Now.ToString('yyyy-MM-dd_HH-mm-ss') # Saving DateTime in predefined format. This is required, since we don't know the target localization settings and want to make sure that UI and engine can communicate in same language.
$Run_Status.engineversion = $EngineVersion;
$Run_Status.targetcomputer =  $Env:ComputerName;

If ($m_OSBuild.Length -gt 0) {
    $Run_Status.targetos = $m_OSName + " build " + $m_OSBuild;
} Else {
    $Run_Status.targetos = $m_OSName;
}

$PackDefinitionXml.Save($ResultsXml);

# Create new variables for counting of successful/failed/skipped entries execution. This is used in run_status reporting.
$Run_Status.entries_total = $PackDefinitionXml.SelectNodes("//entry").Count.ToString();
[Int]$Run_Status_Success = 0;
[Int]$Run_Status_Failed = 0;

# Add CTXOE modules to PSModulePath variable. With this modules can be loaded dynamically based on the prefix.
Write-Host "Adding CTXOE modules"
$Global:CTXOE_Modules = "$CTXOE_Main\Modules"
$Env:PSModulePath = "$([Environment]::GetEnvironmentVariable("PSModulePath"));$($Global:CTXOE_Modules)"

# Older version of PowerShell cannot load modules on-demand. All modules are pre-loaded.
If ($Host.Version.Major -le 2) {
    Write-Host "Detected older version of PowerShell. Importing all modules manually."
    ForEach ($m_Module in $(Get-ChildItem -Path "$CTXOE_Main\Modules" -Recurse -Filter "*.psm1")) {
        Import-Module -Name $m_Module.FullName
    }
}

# If mode is rollback, check if definition file contains the required history elements
If ($PluginMode -eq "Rollback") {
    If ($PackDefinitionXml.SelectNodes("//rollbackparams").Count -eq 0) {
        Throw "You need to select a log file from execution for rollback. This is usually called execute_history.xml. The file specified doesn't include instructions for rollback"
    }
}

# Display metadata for selected template. This acts as a header information about template
$PackDefinitionXml.root.metadata.ChildNodes | Select-Object Name, InnerText | Format-Table -HideTableHeaders

# First version of templates organized groups in packs. This was never really used and < pack/> element was removed in schema version 2.0
# This code is used for backwards compatibility with older templates
If ($PackDefinitionXml.root.pack -is [System.Xml.XmlElement]) {
    Write-host "Old template format has been detected, you should migrate to newer format" -for Red;
    $GroupElements = $PackDefinitionXml.SelectNodes("/root/pack/group");
} Else {
    $GroupElements = $PackDefinitionXml.SelectNodes("/root/group");
}

# Check if template has any conditions to process. In rollback mode, we do not need to process conditions - they've been already resolved to $True in execute mode and we should be able to rollback all changes.
If ($PluginMode -ne "rollback" -and -not $IgnoreConditions -and $PackDefinitionXml.root.condition -is [Object]) {
    Write-Host
    Write-Host "Template condition detected"
    [Hashtable]$m_TemplateConditionResult = CTXOE\Test-CTXOECondition -Element $PackDefinitionXml.root.condition; 
    Write-Host "Template condition result: $($m_TemplateConditionResult.Result)"
    Write-Host "Template condition details: $($m_TemplateConditionResult.Details)"
    Write-Host
    If ($m_TemplateConditionResult.Result -eq $False) {
        $Run_Status.run_details = "Execution stopped by template condition: $($m_TemplateConditionResult.Details)";
    }
}

# Check if template supports requested mode. If not, abort execution and throw an error
If ($PackDefinitionXml.root.metadata.SelectSingleNode("$($PluginMode.ToLower())_not_supported") -is [System.Xml.XmlElement]) {
    $Run_Status.run_details = "This template does NOT support requested mode $($PluginMode)!";
}


# Process template
ForEach ($m_Group in $GroupElements) {
    Write-Host
    Write-Host "        Group: $($m_Group.DisplayName)"
    Write-Host "        Group ID: $($m_Group.ID)"

    # Proceed only if the current run status message has NOT been modified. This is used to detect scenarios where template is reporting that run should not proceed, e.g. when conditions are used or mode is unsupported.
    If ($Run_Status.run_details -ne $Run_Status_Default_Message) {
        Write-Host "        Template processing failed, skipping"
        Continue
    }

    If ($Groups.Count -gt 0 -and $Groups -notcontains $m_Group.ID) {
        Write-Host "        Group not included in the -Groups argument, skipping"
        Continue
    }

    If ($m_Group.Enabled -eq "0") {
        Write-Host "    This group is disabled, skipping" -ForegroundColor DarkGray
        Continue
    }
    
    # Check if group supports requested mode. If not, move to next group
    [Boolean]$m_GroupModeNotSupported = $m_Group.SelectSingleNode("$($PluginMode.ToLower())_not_supported") -is [System.Xml.XmlElement]

    If ($m_GroupModeNotSupported) {
        Write-Host "    This group does not support $($PluginMode.ToLower()) mode, skipping" -ForegroundColor DarkGray
    }

    # PowerShell does not have concept of loop scope. We need to clear all variables from previous group before we process next group.
    Remove-Variable m_GroupConditionResult -ErrorAction SilentlyContinue;

    # Check if group has any conditions to process. 
    If ($PluginMode -ne "rollback" -and -not $IgnoreConditions -and $m_Group.condition -is [Object]) {
        Write-Host
        Write-Host "        Group condition detected"
        [Hashtable]$m_GroupConditionResult = CTXOE\Test-CTXOECondition -Element $m_Group.condition; 
        Write-Host "        Group condition result: $($m_GroupConditionResult.Result)"
        Write-Host "        Group condition details: $($m_GroupConditionResult.Details)"
        Write-Host
    }

    ForEach ($m_Entry in $m_Group.SelectNodes("./entry")) {
        Write-Host "            $($m_Entry.Name) - " -NoNewline

        If ($m_Entry.Enabled -eq "0") {
            Write-Host "    This entry is disabled, skipping" -ForegroundColor DarkGray
            CTXOE\New-CTXOEHistoryElement -Element $m_Entry -SystemChanged $False -StartTime $([DateTime]::Now) -Result $False -Details "Entry is disabled"
            Continue
        }

        If ($m_Entry.Execute -eq "0") {
            Write-Host " Entry is not marked for execution, skipping" -ForegroundColor DarkGray
            CTXOE\New-CTXOEHistoryElement -Element $m_Entry -SystemChanged $False -StartTime $([DateTime]::Now) -Result $False -Details "Entry is not marked for execution, skipping"
            Continue
        }

        # Check if entry supports requested mode. If not, move to next entry. If parent group does not support this mode, all entries should be skipped.
        # We need to make sure that ALL entries that are skipped have "Execute" set to 0 - otherwise status summary will fail to properly determine if script run was successful or not
        If (($m_GroupModeNotSupported -eq $True) -or ($m_Entry.SelectSingleNode("$($PluginMode.ToLower())_not_supported") -is [System.Xml.XmlElement])) {
            Write-Host "    This entry does not support $($PluginMode.ToLower()) mode, skipping" -ForegroundColor DarkGray
            CTXOE\New-CTXOEHistoryElement -Element $m_Entry -SystemChanged $False -StartTime $([DateTime]::Now) -Result $False -Details "This entry does not support $($PluginMode.ToLower()) mode, skipping"
            $m_Entry.Execute = "0";
            Continue
        }

        # Check if entry supports requested mode. If not, move to next entry
        If ($m_Entry.SelectSingleNode("$($PluginMode.ToLower())_not_supported") -is [System.Xml.XmlElement]) {
            Write-Host "    This entry does not support $($PluginMode.ToLower()) mode, skipping" -ForegroundColor DarkGray
            CTXOE\New-CTXOEHistoryElement -Element $m_Entry -SystemChanged $False -StartTime $([DateTime]::Now) -Result $False -Details "This entry does not support $($PluginMode.ToLower()) mode, skipping"

            Continue
        }

        # Section to process (parent) group conditions and entry conditions. Can be skipped if -IgnoreConditions is used or in rollback mode
        If ($PluginMode -ne "Rollback" -and -not $IgnoreConditions ) {
            # Check if the group condition has failed. If yes, none of the entries should be processed
            If ($m_GroupConditionResult -is [object] -and $m_GroupConditionResult.Result -eq $False) {
                Write-Host "    This entry is disabled by group condition, skipping" -ForegroundColor DarkGray;
                CTXOE\New-CTXOEHistoryElement -Element $m_Entry -SystemChanged $False -StartTime $([DateTime]::Now) -Result $False -Details "FILTERED: $($m_GroupConditionResult.Details)";
                $m_Entry.Execute = "0";
                Continue
            }

            # PowerShell does not have concept of loop scope. We need to clear all variables from previous group before we process next group.
            Remove-Variable m_ItemConditionResult -ErrorAction SilentlyContinue;

            # Check if this item has any conditions to process. 
            If ($m_Entry.condition -is [Object]) {
                Write-Host
                Write-Host "            Entry condition detected"
                [Hashtable]$m_ItemConditionResult = CTXOE\Test-CTXOECondition -Element $m_Entry.condition; 
                Write-Host "            Entry condition result: $($m_ItemConditionResult.Result)"
                Write-Host "            Entry condition details: $($m_ItemConditionResult.Details)"
                Write-Host
                
                If ($m_ItemConditionResult.Result -eq $False) {
                    Write-Host "    This entry is disabled by condition, skipping" -ForegroundColor DarkGray;
                    CTXOE\New-CTXOEHistoryElement -Element $m_Entry -SystemChanged $False -StartTime $([DateTime]::Now) -Result $False -Details "FILTERED: $($m_ItemConditionResult.Details)";
                    $m_Entry.Execute = "0";
                    Continue;
                }
            }
        }

        $m_Action = $m_Entry.SelectSingleNode("./action")
        Write-Verbose "            Plugin: $($m_Action.Plugin)"

        # While some plugins can use only a single set of instructions to perform all the different operations (typically services or registry keys), this might not be always possible.

        # Good example is "PowerShell" plugin - different code can be used to analyze the action and execute the action (compare "Get-CurrentState -eq $True" for analyze to "Set-CurrentState -Mode Example -Setup Mode1" for execute mode).

        # In order to support this scenarios, it is possible to override the default <params /> element with a custom element for analyze and rollback phases. Default is still <params />. With this implementation, there can be an action that will implement all three elements (analyzeparams, rollbackparams and executeparams).

        [String]$m_ParamsElementName = "params"
        [String]$m_OverrideElement = "$($PluginMode.ToLower())$m_ParamsElementName"

        If ($m_Action.$m_OverrideElement -is [Object]) {
            Write-Verbose "Using custom <$($m_OverrideElement) /> element"
            $m_ParamsElementName = $m_OverrideElement
        }

        # To prevent any unexpected damage to the system, Rollback mode requires use of custom params object and cannot use the default one.
        If ($PluginMode -eq "Rollback" -and $m_Action.$m_OverrideElement -isnot [Object]) {
            If ($m_Entry.history.systemchanged -eq "0") {
                Write-Host "This entry has not changed, skip" -ForegroundColor DarkGray
                Continue
            } Else {
                Write-Host "Rollback mode requires custom instructions that are not available, skip" -ForegroundColor DarkGray
                Continue
            }
        }

        # Reset variables that are used to report the status
        [Boolean]$Global:CTXOE_Result = $False;
        $Global:CTXOE_Details = "No data returned by this entry (this is unexpected)";

        # Two variables used by rollback. First identify that this entry has modified the system. The second should contain information required for rollback of those changes (if possible). This is required only for "execute" mode.
        [Boolean]$Global:CTXOE_SystemChanged = $False

        $Global:CTXOE_ChangeRollbackParams = $Null

        [DateTime]$StartTime = Get-Date;
        CTXOE\Invoke-CTXOEPlugin -PluginName $($m_Action.Plugin) -Params $m_Action.$m_ParamsElementName -Mode $PluginMode -Verbose

        # Test if there is custom details message for current mode or general custom message. This allows you to display friendly message instead of generic error.
        # This can be either mode-specific or generic (message_analyze_true or message_true). Last token (true/false) is used to identify if custom message should be displayed for success or failure
        # If custom message is detected, output from previous function is ignored and CTXOE_Details is replaced with custom text
        [string]$m_OverrideOutputMessageMode = "message_$($PluginMode.ToLower())_$($Global:CTXOE_Result.ToString().ToLower())";
        [string]$m_OverrideOutputMessageGeneric = "message_$($Global:CTXOE_Result.ToString().ToLower())";

        If ($m_Entry.$m_OverrideOutputMessageMode -is [Object]) {
            $Global:CTXOE_Details = $($m_Entry.$m_OverrideOutputMessageMode);
        } ElseIf ($m_Entry.$m_OverrideOutputMessageGeneric -is [Object]) {
            $Global:CTXOE_Details = $($m_Entry.$m_OverrideOutputMessageGeneric);
        }

		# This code is added to have a situation where CTXOE_Result is set, but not to boolean value (for example to empty string). This will prevent engine from crashing and report which entry does not behave as expected.
		# We do this check here so following code does not need to check if returned value exists
		If ($Global:CTXOE_Result -isnot [Boolean]) {
			$Global:CTXOE_Result = $false;
			$Global:CTXOE_Details = "While processing $($m_Entry.Name) from group $($m_Group.ID), there was an error or code did not return expected result. This value should be boolean, while returned value is $($Global:CTXOE_Result.GetType().FullName)."; 
		}

        If ($Global:CTXOE_Result -eq $false) {
            $Run_Status_Failed += 1;
            Write-Host -ForegroundColor Red $CTXOE_Details
        } Else {
            $Run_Status_Success += 1;
            Write-Host -ForegroundColor Green $CTXOE_Details
        }

        # Save information about changes as an element
        CTXOE\New-CTXOEHistoryElement -Element $m_Entry -SystemChanged $CTXOE_SystemChanged -StartTime $StartTime -Result $CTXOE_Result -Details $CTXOE_Details -RollbackInstructions $CTXOE_ChangeRollbackParams

        If ($OptimizerUI) {
            $history = New-Object -TypeName PSObject
            $history.PSObject.TypeNames.Insert(0,"history")
            $history | Add-Member -MemberType NoteProperty -Name GroupID -Value $m_Group.ID
            $history | Add-Member -MemberType NoteProperty -Name EntryName -Value $m_Entry.Name
            $history | Add-Member -MemberType NoteProperty -Name SystemChanged -Value $m_Entry.SystemChanged
            $history | Add-Member -MemberType NoteProperty -Name StartTime -Value $m_Entry.History.StartTime
            $history | Add-Member -MemberType NoteProperty -Name EndTime -Value $m_Entry.History.EndTime
            $history | Add-Member -MemberType NoteProperty -Name Result -Value $m_Entry.History.Return.Result
            $history | Add-Member -MemberType NoteProperty -Name Details -Value $m_Entry.History.Return.Details

            Write-Output $history
        }
    }
}

#Region "Run status processing"
# Finish processing of run_status, save everything to return XML file
$Run_Status.time_end = [DateTime]::Now.ToString('yyyy-MM-dd_HH-mm-ss') # Saving DateTime in predefined format. This is required, since we don't know the target localization settings and want to make sure that UI and engine can communicate in same language.

$Run_Status.entries_success = $Run_Status_Success.ToString();
$Run_Status.entries_failed = $Run_Status_Failed.ToString();

# Run status should be determined ONLY if template has not aborted execution before.
If ($Run_Status.run_details -eq $Run_Status_Default_Message) {
    
    # Count all entries that were expected to execute (execute=1), but have not finished successfully (result!=1)
    [Int]$m_EntriesNotExecuted = $PackDefinitionXml.SelectNodes("//entry[execute=1 and not(history/return/result=1)]").Count

    # If we have entries that are not successful
    If ($m_EntriesNotExecuted -gt 0) {
        If ($m_EntriesNotExecuted -eq 1) {
            $Run_Status.run_details = "$m_EntriesNotExecuted entry has failed";
        } Else {
            $Run_Status.run_details = "$m_EntriesNotExecuted entries have failed";
        }
    # If anything is marked as failed
    } ElseIf ($Run_Status_Failed -gt 0) {
        If ($Run_Status_Failed -eq 1) {
            $Run_Status.run_details = "$Run_Status_Failed entry from this template failed";
        } Else {
            $Run_Status.run_details = "$Run_Status_Failed entries from this template failed";
        }
    # If nothing was actually executed
    } ElseIf ($Run_Status_Success -eq 0) {
        $Run_Status.run_details = "No entries from this template have been processed";
    # Nothing failed, something was successful = sounds good
    } ElseIf ($Run_Status_Success -gt 0 -and $Run_Status_Failed -eq 0) {
        $Run_Status.run_successful = $True.ToString();
        $Run_Status.run_details = "Template has been processed successfully";
    } Else {
        $Run_Status.run_details = "Unknown condition when evaluating run result";  
    }
}
#EndRegion

# Send the overall execute result for UI to show
If ($OptimizerUI) {
    $overallresult = New-Object -TypeName PSObject
    $overallresult.PSObject.TypeNames.Insert(0,"overallresult")
    $overallresult | Add-Member -MemberType NoteProperty -Name run_successful -Value $Run_Status.run_successful
    $overallresult | Add-Member -MemberType NoteProperty -Name run_details -Value $Run_Status.run_details
    $overallresult | Add-Member -MemberType NoteProperty -Name entries_success -Value $Run_Status.entries_success
    $overallresult | Add-Member -MemberType NoteProperty -Name entries_failed -Value $Run_Status.entries_failed

    Write-Output $overallresult
}
# end

# Save the output in XML format for further parsing\history
$PackDefinitionXml.Save($ResultsXml);

#Region "Registry status reporting"

# If mode is 'execute', then save registry status. If mode is 'rollback' (and registry status exists), remove it. No action required for 'analyze' mode

[String]$m_RegistryPath = "HKLM:\SOFTWARE\Citrix\Optimizer\" + $PackDefinitionXml.root.metadata.category;

If ($PluginMode -eq "execute") {
    # Check if registry key exists
    If ($(Test-Path $m_RegistryPath) -eq $False) {
        # If registry key doesn't exist, create it
        New-Item -Path $m_RegistryPath -Force | Out-Null;
    }

    # Save location of XML file that contains more details about execution
    New-ItemProperty -Path $m_RegistryPath -Name "log_path" -PropertyType "string" -Value $ResultsXml -Force | Out-Null;

    # Save all <metadata /> and <run_status />
    ForEach ($m_Node in $PackDefinitionXml.root.metadata.SelectNodes("*")) {
        New-ItemProperty -Path $m_RegistryPath -Name $m_Node.Name -PropertyType "string" -Value $m_Node.InnerText -Force | Out-Null;
    }
    ForEach ($m_Node in $PackDefinitionXml.root.run_status.SelectNodes("*")) {
        New-ItemProperty -Path $m_RegistryPath -Name $m_Node.Name -PropertyType "string" -Value $m_Node.InnerText -Force | Out-Null;
    }

} ElseIf ($PluginMode -eq "rollback") {
    # Check if registry key exists
    If ($(Test-Path $m_RegistryPath) -eq $True) {
        # If registry key exists, delete it
        Remove-Item -Path $m_RegistryPath -Force | Out-Null;
    }
}
#EndRegion

# Use transformation file to generate HTML report
$XSLT = New-Object System.Xml.Xsl.XslCompiledTransform;
$XSLT.Load("$CTXOE_Main\CtxOptimizerReport.xslt");
$XSLT.Transform($ResultsXml, $OutputHtml);

# If another location is requested, save the XML file here as well.
If ($OutputXml.Length -gt 0) {
    $PackDefinitionXml.Save($OutputXml);
}

# If the current host is transcribing, save the transcription
Try {
    Stop-Transcript | Out-Null
} Catch { Write-Host "An exception happened when stopping transcription: $_" -ForegroundColor Red }

# SIG # Begin signature block
# MIIoogYJKoZIhvcNAQcCoIIokzCCKI8CAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCAEQ19w+DJ9tWYA
# rPE+d1bjxCgCYDnU1tMAtwlowHS5IKCCDcAwggawMIIEmKADAgECAhAIrUCyYNKc
# TJ9ezam9k67ZMA0GCSqGSIb3DQEBDAUAMGIxCzAJBgNVBAYTAlVTMRUwEwYDVQQK
# EwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xITAfBgNV
# BAMTGERpZ2lDZXJ0IFRydXN0ZWQgUm9vdCBHNDAeFw0yMTA0MjkwMDAwMDBaFw0z
# NjA0MjgyMzU5NTlaMGkxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwg
# SW5jLjFBMD8GA1UEAxM4RGlnaUNlcnQgVHJ1c3RlZCBHNCBDb2RlIFNpZ25pbmcg
# UlNBNDA5NiBTSEEzODQgMjAyMSBDQTEwggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAw
# ggIKAoICAQDVtC9C0CiteLdd1TlZG7GIQvUzjOs9gZdwxbvEhSYwn6SOaNhc9es0
# JAfhS0/TeEP0F9ce2vnS1WcaUk8OoVf8iJnBkcyBAz5NcCRks43iCH00fUyAVxJr
# Q5qZ8sU7H/Lvy0daE6ZMswEgJfMQ04uy+wjwiuCdCcBlp/qYgEk1hz1RGeiQIXhF
# LqGfLOEYwhrMxe6TSXBCMo/7xuoc82VokaJNTIIRSFJo3hC9FFdd6BgTZcV/sk+F
# LEikVoQ11vkunKoAFdE3/hoGlMJ8yOobMubKwvSnowMOdKWvObarYBLj6Na59zHh
# 3K3kGKDYwSNHR7OhD26jq22YBoMbt2pnLdK9RBqSEIGPsDsJ18ebMlrC/2pgVItJ
# wZPt4bRc4G/rJvmM1bL5OBDm6s6R9b7T+2+TYTRcvJNFKIM2KmYoX7BzzosmJQay
# g9Rc9hUZTO1i4F4z8ujo7AqnsAMrkbI2eb73rQgedaZlzLvjSFDzd5Ea/ttQokbI
# YViY9XwCFjyDKK05huzUtw1T0PhH5nUwjewwk3YUpltLXXRhTT8SkXbev1jLchAp
# QfDVxW0mdmgRQRNYmtwmKwH0iU1Z23jPgUo+QEdfyYFQc4UQIyFZYIpkVMHMIRro
# OBl8ZhzNeDhFMJlP/2NPTLuqDQhTQXxYPUez+rbsjDIJAsxsPAxWEQIDAQABo4IB
# WTCCAVUwEgYDVR0TAQH/BAgwBgEB/wIBADAdBgNVHQ4EFgQUaDfg67Y7+F8Rhvv+
# YXsIiGX0TkIwHwYDVR0jBBgwFoAU7NfjgtJxXWRM3y5nP+e6mK4cD08wDgYDVR0P
# AQH/BAQDAgGGMBMGA1UdJQQMMAoGCCsGAQUFBwMDMHcGCCsGAQUFBwEBBGswaTAk
# BggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMEEGCCsGAQUFBzAC
# hjVodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkUm9v
# dEc0LmNydDBDBgNVHR8EPDA6MDigNqA0hjJodHRwOi8vY3JsMy5kaWdpY2VydC5j
# b20vRGlnaUNlcnRUcnVzdGVkUm9vdEc0LmNybDAcBgNVHSAEFTATMAcGBWeBDAED
# MAgGBmeBDAEEATANBgkqhkiG9w0BAQwFAAOCAgEAOiNEPY0Idu6PvDqZ01bgAhql
# +Eg08yy25nRm95RysQDKr2wwJxMSnpBEn0v9nqN8JtU3vDpdSG2V1T9J9Ce7FoFF
# UP2cvbaF4HZ+N3HLIvdaqpDP9ZNq4+sg0dVQeYiaiorBtr2hSBh+3NiAGhEZGM1h
# mYFW9snjdufE5BtfQ/g+lP92OT2e1JnPSt0o618moZVYSNUa/tcnP/2Q0XaG3Ryw
# YFzzDaju4ImhvTnhOE7abrs2nfvlIVNaw8rpavGiPttDuDPITzgUkpn13c5Ubdld
# AhQfQDN8A+KVssIhdXNSy0bYxDQcoqVLjc1vdjcshT8azibpGL6QB7BDf5WIIIJw
# 8MzK7/0pNVwfiThV9zeKiwmhywvpMRr/LhlcOXHhvpynCgbWJme3kuZOX956rEnP
# LqR0kq3bPKSchh/jwVYbKyP/j7XqiHtwa+aguv06P0WmxOgWkVKLQcBIhEuWTatE
# QOON8BUozu3xGFYHKi8QxAwIZDwzj64ojDzLj4gLDb879M4ee47vtevLt/B3E+bn
# KD+sEq6lLyJsQfmCXBVmzGwOysWGw/YmMwwHS6DTBwJqakAwSEs0qFEgu60bhQji
# WQ1tygVQK+pKHJ6l/aCnHwZ05/LWUpD9r4VIIflXO7ScA+2GRfS0YW6/aOImYIbq
# yK+p/pQd52MbOoZWeE4wggcIMIIE8KADAgECAhADWiAGLDpbT4qELERZjaX6MA0G
# CSqGSIb3DQEBCwUAMGkxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwg
# SW5jLjFBMD8GA1UEAxM4RGlnaUNlcnQgVHJ1c3RlZCBHNCBDb2RlIFNpZ25pbmcg
# UlNBNDA5NiBTSEEzODQgMjAyMSBDQTEwHhcNMjUwMjA3MDAwMDAwWhcNMjYwMjA2
# MjM1OTU5WjCBjzELMAkGA1UEBhMCVVMxEDAOBgNVBAgTB0Zsb3JpZGExGDAWBgNV
# BAcTD0ZvcnQgTGF1ZGVyZGFsZTEdMBsGA1UEChMUQ2l0cml4IFN5c3RlbXMsIElu
# Yy4xFjAUBgNVBAsTDUNpdHJpeCBYZW5BcHAxHTAbBgNVBAMTFENpdHJpeCBTeXN0
# ZW1zLCBJbmMuMIIBojANBgkqhkiG9w0BAQEFAAOCAY8AMIIBigKCAYEAnBm4JMW+
# W2tRIJltyaJM/q5G6WRAN293DlUc71U/dg6HzXRtahZP2kvV6ii32orbgdNkCwig
# +bo0W7AKjBBHKJqP96nuTNTsHoz+aeEE6PfgaFzGTRq77+dGDCk5DuPbFaXJCn9s
# UjVdbJTG1YXAcll8ZU6Iarg3eOokp2CcejMcjeD2dO8y89o6y2W5sWj2oIA+QRE9
# iU5qNfDLUtXo5i017JTN+qs9RtuwD77svXoV29wmkPmGuUq625ZZYFtRa0/t/C7w
# /k00UOFjykbvNBPj/cT67i/J/Um8tOanuC3cYU18VToMsfpS2t4irTdtrBzHupr/
# MB2DzZTwr5x2m3UzgEsnrr9bYnCIpOuW+K/oExuTpHtZsk6fnpoteOfyP059dNMg
# i1Gj074k6JfaJG+6fwKW0i2Unf7NDBArHdoHA6eIYB/OivPt4cmusgzRr2PziAl4
# LpA/9VRcnR68CWjnoTTr7qhdDeGqMgk24cNtmg+6BDt65GDVDX2ycthpAgMBAAGj
# ggIDMIIB/zAfBgNVHSMEGDAWgBRoN+Drtjv4XxGG+/5hewiIZfROQjAdBgNVHQ4E
# FgQUEpWeJpU+v9Nt97sQvlQgm4DN5wMwPgYDVR0gBDcwNTAzBgZngQwBBAEwKTAn
# BggrBgEFBQcCARYbaHR0cDovL3d3dy5kaWdpY2VydC5jb20vQ1BTMA4GA1UdDwEB
# /wQEAwIHgDATBgNVHSUEDDAKBggrBgEFBQcDAzCBtQYDVR0fBIGtMIGqMFOgUaBP
# hk1odHRwOi8vY3JsMy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkRzRDb2Rl
# U2lnbmluZ1JTQTQwOTZTSEEzODQyMDIxQ0ExLmNybDBToFGgT4ZNaHR0cDovL2Ny
# bDQuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZEc0Q29kZVNpZ25pbmdSU0E0
# MDk2U0hBMzg0MjAyMUNBMS5jcmwwgZQGCCsGAQUFBwEBBIGHMIGEMCQGCCsGAQUF
# BzABhhhodHRwOi8vb2NzcC5kaWdpY2VydC5jb20wXAYIKwYBBQUHMAKGUGh0dHA6
# Ly9jYWNlcnRzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFRydXN0ZWRHNENvZGVTaWdu
# aW5nUlNBNDA5NlNIQTM4NDIwMjFDQTEuY3J0MAkGA1UdEwQCMAAwDQYJKoZIhvcN
# AQELBQADggIBAMGu7yYWlIS96uBwpNvMWgRgVFRwDx0HBdxGRi2+XvrtKkooOglx
# eUl/IYzZb8WhEBs/vknAmZkxBnkbbdgKKqxDeIkT2JX1sNGLKIzBBLPPjGMd2wxH
# CAVssWrA9xMESJ5Zx+5jHyoeVPt4XdPAXNB6Ca1JCa68Ip8Bn+atqop5wOmhXK85
# 2V2PdCWXjQr3Yl0k0PY/OtDDxnQc0dClj9TYlfGsOwyVNNQh/eExH2wkVlYYxgcj
# Zn89SmO4L+9AHkfpj/wd+/pJ+BYSflAotC6d9l4dft6f6PZ6KkbAz56dVkBX20xz
# YKKIK9tCf5sa1wnjkBEm2yn1IDn6K1cwUGO0whhtzGgcv4vPfv5bkumCSD7AquQv
# 37LP/gOZT9v3CiZAlTbZAoOK105qFmawt7AKPFzNFq9EjbUHYVBeYFnETDoa9zLz
# KCCcF/xJsjfn7YqFM+b1zq3C1YJK1B6+f3b5acpXNHbyMxPSOeNDZ13wMAYKah17
# D/4GZe74bsVqFo0oHlB82Bocr0AnnrLnLvfEirc0KIon5CQ5cci3LolBE0aNhD+n
# UtInPAJN1uRFpvhSK1ucVO/afmTQVtUgp5a8a8Dthc0bqh2ere4hoY+9U08Bda5g
# mFOO9YU8xEU3Do0AbuJPUVVALAAs0I+38RjYtIfPPt/T1zrMFHv+CMsKMYIaODCC
# GjQCAQEwfTBpMQswCQYDVQQGEwJVUzEXMBUGA1UEChMORGlnaUNlcnQsIEluYy4x
# QTA/BgNVBAMTOERpZ2lDZXJ0IFRydXN0ZWQgRzQgQ29kZSBTaWduaW5nIFJTQTQw
# OTYgU0hBMzg0IDIwMjEgQ0ExAhADWiAGLDpbT4qELERZjaX6MA0GCWCGSAFlAwQC
# AQUAoIHQMBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3AgEEMBwGCisGAQQBgjcCAQsx
# DjAMBgorBgEEAYI3AgEVMC8GCSqGSIb3DQEJBDEiBCAeu5XMgOYfcmU2tR3obwEj
# E5ZMba3xf5wXVuOemV1zSzBkBgorBgEEAYI3AgEMMVYwVKA4gDYAQwBpAHQAcgBp
# AHgAIABTAHUAcABwAG8AcgB0AGEAYgBpAGwAaQB0AHkAIABUAG8AbwBsAHOhGIAW
# aHR0cDovL3d3dy5jaXRyaXguY29tIDANBgkqhkiG9w0BAQEFAASCAYACUMHE+7yV
# 9JG5y6O/Ng98SS8jWGiJTkgpiHgNE8WoMuOAxyVujnhHxJC4kpa3jV/5nSi0dK1U
# +qm+8u1BP4URrF+kCAkVrwnPGo4doGC6FzMXEu7Oisp07WScJRfoZB0fnnC+3v/8
# DMRgq6nnNVFcvqC0PbxdziePYOTAmpkXEeLykV+T1t3gSG6WuKWAfgYrJy+TZqd+
# S0TFtuQEyZNolFDdTeC5B+IBXClUj/h6m/gq5JyIFG94TPvJwdtUo8X6A1XHHBbv
# /DJT9ncBnfA+HYrzgXzkmWlGqKnockWtP0IWG+lZi7hSBNvtRlmgC8gHqmOa7+3A
# yFHSI63ACeErRzz6aS60aODhUDM9GTPVyrHSVRnvEtnvMFvQmUkT4Ha0g/z24GuG
# MGCsw9OcOZxrXTeEXGejV9/sOQ8rF73qLsdGv9/xl132p77W+XeY6/0NO/wzO/jb
# A45u2V2jBHkQSlcg8Dcze71itFYJJYfSox+VqQOmGDxQYiSb4zUptdGhghc5MIIX
# NQYKKwYBBAGCNwMDATGCFyUwghchBgkqhkiG9w0BBwKgghcSMIIXDgIBAzEPMA0G
# CWCGSAFlAwQCAQUAMHcGCyqGSIb3DQEJEAEEoGgEZjBkAgEBBglghkgBhv1sBwEw
# MTANBglghkgBZQMEAgEFAAQgN9g8cKr00XW6hGmeMLfoMnX8M1PfyjMe7jGSMq0O
# AyACEDUooSBoV4QfJmm6UgTd14cYDzIwMjUwMzExMDYyNTA1WqCCEwMwgga8MIIE
# pKADAgECAhALrma8Wrp/lYfG+ekE4zMEMA0GCSqGSIb3DQEBCwUAMGMxCzAJBgNV
# BAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwgSW5jLjE7MDkGA1UEAxMyRGlnaUNl
# cnQgVHJ1c3RlZCBHNCBSU0E0MDk2IFNIQTI1NiBUaW1lU3RhbXBpbmcgQ0EwHhcN
# MjQwOTI2MDAwMDAwWhcNMzUxMTI1MjM1OTU5WjBCMQswCQYDVQQGEwJVUzERMA8G
# A1UEChMIRGlnaUNlcnQxIDAeBgNVBAMTF0RpZ2lDZXJ0IFRpbWVzdGFtcCAyMDI0
# MIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEAvmpzn/aVIauWMLpbbeZZ
# o7Xo/ZEfGMSIO2qZ46XB/QowIEMSvgjEdEZ3v4vrrTHleW1JWGErrjOL0J4L0HqV
# R1czSzvUQ5xF7z4IQmn7dHY7yijvoQ7ujm0u6yXF2v1CrzZopykD07/9fpAT4Bxp
# T9vJoJqAsP8YuhRvflJ9YeHjes4fduksTHulntq9WelRWY++TFPxzZrbILRYynyE
# y7rS1lHQKFpXvo2GePfsMRhNf1F41nyEg5h7iOXv+vjX0K8RhUisfqw3TTLHj1uh
# S66YX2LZPxS4oaf33rp9HlfqSBePejlYeEdU740GKQM7SaVSH3TbBL8R6HwX9QVp
# GnXPlKdE4fBIn5BBFnV+KwPxRNUNK6lYk2y1WSKour4hJN0SMkoaNV8hyyADiX1x
# uTxKaXN12HgR+8WulU2d6zhzXomJ2PleI9V2yfmfXSPGYanGgxzqI+ShoOGLomMd
# 3mJt92nm7Mheng/TBeSA2z4I78JpwGpTRHiT7yHqBiV2ngUIyCtd0pZ8zg3S7bk4
# QC4RrcnKJ3FbjyPAGogmoiZ33c1HG93Vp6lJ415ERcC7bFQMRbxqrMVANiav1k42
# 5zYyFMyLNyE1QulQSgDpW9rtvVcIH7WvG9sqYup9j8z9J1XqbBZPJ5XLln8mS8wW
# mdDLnBHXgYly/p1DhoQo5fkCAwEAAaOCAYswggGHMA4GA1UdDwEB/wQEAwIHgDAM
# BgNVHRMBAf8EAjAAMBYGA1UdJQEB/wQMMAoGCCsGAQUFBwMIMCAGA1UdIAQZMBcw
# CAYGZ4EMAQQCMAsGCWCGSAGG/WwHATAfBgNVHSMEGDAWgBS6FtltTYUvcyl2mi91
# jGogj57IbzAdBgNVHQ4EFgQUn1csA3cOKBWQZqVjXu5Pkh92oFswWgYDVR0fBFMw
# UTBPoE2gS4ZJaHR0cDovL2NybDMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3Rl
# ZEc0UlNBNDA5NlNIQTI1NlRpbWVTdGFtcGluZ0NBLmNybDCBkAYIKwYBBQUHAQEE
# gYMwgYAwJAYIKwYBBQUHMAGGGGh0dHA6Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBYBggr
# BgEFBQcwAoZMaHR0cDovL2NhY2VydHMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0VHJ1
# c3RlZEc0UlNBNDA5NlNIQTI1NlRpbWVTdGFtcGluZ0NBLmNydDANBgkqhkiG9w0B
# AQsFAAOCAgEAPa0eH3aZW+M4hBJH2UOR9hHbm04IHdEoT8/T3HuBSyZeq3jSi5GX
# eWP7xCKhVireKCnCs+8GZl2uVYFvQe+pPTScVJeCZSsMo1JCoZN2mMew/L4tpqVN
# bSpWO9QGFwfMEy60HofN6V51sMLMXNTLfhVqs+e8haupWiArSozyAmGH/6oMQAh0
# 78qRh6wvJNU6gnh5OruCP1QUAvVSu4kqVOcJVozZR5RRb/zPd++PGE3qF1P3xWvY
# ViUJLsxtvge/mzA75oBfFZSbdakHJe2BVDGIGVNVjOp8sNt70+kEoMF+T6tptMUN
# lehSR7vM+C13v9+9ZOUKzfRUAYSyyEmYtsnpltD/GWX8eM70ls1V6QG/ZOB6b6Yu
# m1HvIiulqJ1Elesj5TMHq8CWT/xrW7twipXTJ5/i5pkU5E16RSBAdOp12aw8IQhh
# A/vEbFkEiF2abhuFixUDobZaA0VhqAsMHOmaT3XThZDNi5U2zHKhUs5uHHdG6BoQ
# au75KiNbh0c+hatSF+02kULkftARjsyEpHKsF7u5zKRbt5oK5YGwFvgc4pEVUNyt
# mB3BpIiowOIIuDgP5M9WArHYSAR16gc0dP2XdkMEP5eBsX7bf/MGN4K3HP50v/01
# ZHo/Z5lGLvNwQ7XHBx1yomzLP8lx4Q1zZKDyHcp4VQJLu2kWTsKsOqQwggauMIIE
# lqADAgECAhAHNje3JFR82Ees/ShmKl5bMA0GCSqGSIb3DQEBCwUAMGIxCzAJBgNV
# BAYTAlVTMRUwEwYDVQQKEwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdp
# Y2VydC5jb20xITAfBgNVBAMTGERpZ2lDZXJ0IFRydXN0ZWQgUm9vdCBHNDAeFw0y
# MjAzMjMwMDAwMDBaFw0zNzAzMjIyMzU5NTlaMGMxCzAJBgNVBAYTAlVTMRcwFQYD
# VQQKEw5EaWdpQ2VydCwgSW5jLjE7MDkGA1UEAxMyRGlnaUNlcnQgVHJ1c3RlZCBH
# NCBSU0E0MDk2IFNIQTI1NiBUaW1lU3RhbXBpbmcgQ0EwggIiMA0GCSqGSIb3DQEB
# AQUAA4ICDwAwggIKAoICAQDGhjUGSbPBPXJJUVXHJQPE8pE3qZdRodbSg9GeTKJt
# oLDMg/la9hGhRBVCX6SI82j6ffOciQt/nR+eDzMfUBMLJnOWbfhXqAJ9/UO0hNoR
# 8XOxs+4rgISKIhjf69o9xBd/qxkrPkLcZ47qUT3w1lbU5ygt69OxtXXnHwZljZQp
# 09nsad/ZkIdGAHvbREGJ3HxqV3rwN3mfXazL6IRktFLydkf3YYMZ3V+0VAshaG43
# IbtArF+y3kp9zvU5EmfvDqVjbOSmxR3NNg1c1eYbqMFkdECnwHLFuk4fsbVYTXn+
# 149zk6wsOeKlSNbwsDETqVcplicu9Yemj052FVUmcJgmf6AaRyBD40NjgHt1bicl
# kJg6OBGz9vae5jtb7IHeIhTZgirHkr+g3uM+onP65x9abJTyUpURK1h0QCirc0PO
# 30qhHGs4xSnzyqqWc0Jon7ZGs506o9UD4L/wojzKQtwYSH8UNM/STKvvmz3+Drhk
# Kvp1KCRB7UK/BZxmSVJQ9FHzNklNiyDSLFc1eSuo80VgvCONWPfcYd6T/jnA+bIw
# pUzX6ZhKWD7TA4j+s4/TXkt2ElGTyYwMO1uKIqjBJgj5FBASA31fI7tk42PgpuE+
# 9sJ0sj8eCXbsq11GdeJgo1gJASgADoRU7s7pXcheMBK9Rp6103a50g5rmQzSM7TN
# sQIDAQABo4IBXTCCAVkwEgYDVR0TAQH/BAgwBgEB/wIBADAdBgNVHQ4EFgQUuhbZ
# bU2FL3MpdpovdYxqII+eyG8wHwYDVR0jBBgwFoAU7NfjgtJxXWRM3y5nP+e6mK4c
# D08wDgYDVR0PAQH/BAQDAgGGMBMGA1UdJQQMMAoGCCsGAQUFBwMIMHcGCCsGAQUF
# BwEBBGswaTAkBggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMEEG
# CCsGAQUFBzAChjVodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRU
# cnVzdGVkUm9vdEc0LmNydDBDBgNVHR8EPDA6MDigNqA0hjJodHRwOi8vY3JsMy5k
# aWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkUm9vdEc0LmNybDAgBgNVHSAEGTAX
# MAgGBmeBDAEEAjALBglghkgBhv1sBwEwDQYJKoZIhvcNAQELBQADggIBAH1ZjsCT
# tm+YqUQiAX5m1tghQuGwGC4QTRPPMFPOvxj7x1Bd4ksp+3CKDaopafxpwc8dB+k+
# YMjYC+VcW9dth/qEICU0MWfNthKWb8RQTGIdDAiCqBa9qVbPFXONASIlzpVpP0d3
# +3J0FNf/q0+KLHqrhc1DX+1gtqpPkWaeLJ7giqzl/Yy8ZCaHbJK9nXzQcAp876i8
# dU+6WvepELJd6f8oVInw1YpxdmXazPByoyP6wCeCRK6ZJxurJB4mwbfeKuv2nrF5
# mYGjVoarCkXJ38SNoOeY+/umnXKvxMfBwWpx2cYTgAnEtp/Nh4cku0+jSbl3ZpHx
# cpzpSwJSpzd+k1OsOx0ISQ+UzTl63f8lY5knLD0/a6fxZsNBzU+2QJshIUDQtxMk
# zdwdeDrknq3lNHGS1yZr5Dhzq6YBT70/O3itTK37xJV77QpfMzmHQXh6OOmc4d0j
# /R0o08f56PGYX/sr2H7yRp11LB4nLCbbbxV7HhmLNriT1ObyF5lZynDwN7+YAN8g
# Fk8n+2BnFqFmut1VwDophrCYoCvtlUG3OtUVmDG0YgkPCr2B2RP+v6TR81fZvAT6
# gt4y3wSJ8ADNXcL50CN/AAvkdgIm2fBldkKmKYcJRyvmfxqkhQ/8mJb2VVQrH4D6
# wPIOK+XW+6kvRBVK5xMOHds3OBqhK/bt1nz8MIIFjTCCBHWgAwIBAgIQDpsYjvnQ
# Lefv21DiCEAYWjANBgkqhkiG9w0BAQwFADBlMQswCQYDVQQGEwJVUzEVMBMGA1UE
# ChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQLExB3d3cuZGlnaWNlcnQuY29tMSQwIgYD
# VQQDExtEaWdpQ2VydCBBc3N1cmVkIElEIFJvb3QgQ0EwHhcNMjIwODAxMDAwMDAw
# WhcNMzExMTA5MjM1OTU5WjBiMQswCQYDVQQGEwJVUzEVMBMGA1UEChMMRGlnaUNl
# cnQgSW5jMRkwFwYDVQQLExB3d3cuZGlnaWNlcnQuY29tMSEwHwYDVQQDExhEaWdp
# Q2VydCBUcnVzdGVkIFJvb3QgRzQwggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAwggIK
# AoICAQC/5pBzaN675F1KPDAiMGkz7MKnJS7JIT3yithZwuEppz1Yq3aaza57G4QN
# xDAf8xukOBbrVsaXbR2rsnnyyhHS5F/WBTxSD1Ifxp4VpX6+n6lXFllVcq9ok3DC
# srp1mWpzMpTREEQQLt+C8weE5nQ7bXHiLQwb7iDVySAdYyktzuxeTsiT+CFhmzTr
# BcZe7FsavOvJz82sNEBfsXpm7nfISKhmV1efVFiODCu3T6cw2Vbuyntd463JT17l
# Necxy9qTXtyOj4DatpGYQJB5w3jHtrHEtWoYOAMQjdjUN6QuBX2I9YI+EJFwq1WC
# QTLX2wRzKm6RAXwhTNS8rhsDdV14Ztk6MUSaM0C/CNdaSaTC5qmgZ92kJ7yhTzm1
# EVgX9yRcRo9k98FpiHaYdj1ZXUJ2h4mXaXpI8OCiEhtmmnTK3kse5w5jrubU75KS
# Op493ADkRSWJtppEGSt+wJS00mFt6zPZxd9LBADMfRyVw4/3IbKyEbe7f/LVjHAs
# QWCqsWMYRJUadmJ+9oCw++hkpjPRiQfhvbfmQ6QYuKZ3AeEPlAwhHbJUKSWJbOUO
# UlFHdL4mrLZBdd56rF+NP8m800ERElvlEFDrMcXKchYiCd98THU/Y+whX8QgUWtv
# sauGi0/C1kVfnSD8oR7FwI+isX4KJpn15GkvmB0t9dmpsh3lGwIDAQABo4IBOjCC
# ATYwDwYDVR0TAQH/BAUwAwEB/zAdBgNVHQ4EFgQU7NfjgtJxXWRM3y5nP+e6mK4c
# D08wHwYDVR0jBBgwFoAUReuir/SSy4IxLVGLp6chnfNtyA8wDgYDVR0PAQH/BAQD
# AgGGMHkGCCsGAQUFBwEBBG0wazAkBggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGln
# aWNlcnQuY29tMEMGCCsGAQUFBzAChjdodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5j
# b20vRGlnaUNlcnRBc3N1cmVkSURSb290Q0EuY3J0MEUGA1UdHwQ+MDwwOqA4oDaG
# NGh0dHA6Ly9jcmwzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydEFzc3VyZWRJRFJvb3RD
# QS5jcmwwEQYDVR0gBAowCDAGBgRVHSAAMA0GCSqGSIb3DQEBDAUAA4IBAQBwoL9D
# XFXnOF+go3QbPbYW1/e/Vwe9mqyhhyzshV6pGrsi+IcaaVQi7aSId229GhT0E0p6
# Ly23OO/0/4C5+KH38nLeJLxSA8hO0Cre+i1Wz/n096wwepqLsl7Uz9FDRJtDIeuW
# cqFItJnLnU+nBgMTdydE1Od/6Fmo8L8vC6bp8jQ87PcDx4eo0kxAGTVGamlUsLih
# Vo7spNU96LHc/RzY9HdaXFSMb++hUD38dglohJ9vytsgjTVgHAIDyyCwrFigDkBj
# xZgiwbJZ9VVrzyerbHbObyMt9H5xaiNrIv8SuFQtJ37YOtnwtoeW/VvRXKwYw02f
# c7cBqZ9Xql4o4rmUMYIDdjCCA3ICAQEwdzBjMQswCQYDVQQGEwJVUzEXMBUGA1UE
# ChMORGlnaUNlcnQsIEluYy4xOzA5BgNVBAMTMkRpZ2lDZXJ0IFRydXN0ZWQgRzQg
# UlNBNDA5NiBTSEEyNTYgVGltZVN0YW1waW5nIENBAhALrma8Wrp/lYfG+ekE4zME
# MA0GCWCGSAFlAwQCAQUAoIHRMBoGCSqGSIb3DQEJAzENBgsqhkiG9w0BCRABBDAc
# BgkqhkiG9w0BCQUxDxcNMjUwMzExMDYyNTA1WjArBgsqhkiG9w0BCRACDDEcMBow
# GDAWBBTb04XuYtvSPnvk9nFIUIck1YZbRTAvBgkqhkiG9w0BCQQxIgQgmgIgp968
# oEXuFUgGhqi03dq//GH6bwuquRasKTf4DYQwNwYLKoZIhvcNAQkQAi8xKDAmMCQw
# IgQgdnafqPJjLx9DCzojMK7WVnX+13PbBdZluQWTmEOPmtswDQYJKoZIhvcNAQEB
# BQAEggIAS5QrgZsB0B7TUiqH1j4zpJ/Kt0RePQz4jOGz7Kc/q/u+imexI4A8UB4Q
# DmxrxlF6IOL7FD6//unZ4iKXdZF/Q5oMD2xs9RnG3eOyfYbSuZrP6eDKSgEW0CLb
# 9Q4g7SQFERM3mOxz0csm3ap1ntAnU3jAErZvX+RwEi+R2pX8Cp5rEXOe4ZOxFig0
# Il9QUIF+XUscXrG7LoJKnOECpwd4W6o6T4fekcnOBOIo0kZ/5sPX5qWboYjMy2yT
# W28GX7jzqrtg1Y4P4Y6tKwogx0ZEROoVn3217c5qmNyvXFH7PzRQhom8UsJziwfF
# HT2SUovvLi8EzQRvNYSBX2BUdiu9e078DJUTJFxXujq1O+1MinpbaVLiBmfgzLmo
# QkiTAinNalYFz5q0xmyaN4CKAqmTirjV6RBFem7gRBIui4YEgoIiWYMW/jGL+Q3P
# iNYCEPYsubZSiznTUhYcvLT7aw7/OpfTe07ZEJbHLxnak/ckJ+iGbaTmpSJhnwYq
# 6TyTxaugkW9ROq4GGPVriny6LOC5S4PwSOUmuzySULfsEWfdNQVDcp44kvdtLbFg
# 2EdeLSz+6fEB+aU/D2MEIMKagahF3sC+SKocbHzv718s207+amvvI50VSd/epJAG
# t/NZXm9aEzStWW23e1OOobu3glLuvIEL3k+E66Eu6XXRScFXLns=
# SIG # End signature block
